<?php
include_once "../config.php";
if (session_status() === PHP_SESSION_NONE) { session_start(); }

if (empty($_SESSION['logged'])) { header("Location: login.php"); exit; }

auto_check_page_permission();

// Staff dengan permission view_users hanya bisa READ-ONLY
$is_readonly = !is_master();

// 🔧 INITIALIZE VARIABLES - FIX UNTUK UNDEFINED VARIABLE
$err = '';
$ok = '';

// Handle CRUD hanya jika Master
if (!$is_readonly && $_SERVER['REQUEST_METHOD'] === 'POST') {
  $role = $_SESSION['role'] ?? '';
  if ($role !== 'master') { header('HTTP/1.1 403 Forbidden'); echo 'Forbidden'; exit; }

  // Include log helper
  require_once __DIR__."/includes/log_helper.php";

  // Variable sudah diinisialisasi di atas, tidak perlu lagi:
  // $err = ''; $ok = '';

  function sanitize_username($s){ return preg_replace('/[^a-zA-Z0-9_.-]/', '', trim((string)$s)); }
  function sanitize_iplist($s){
    $s = (string)$s;
    $items = preg_split('/[,\s]+/', $s, -1, PREG_SPLIT_NO_EMPTY);
    $out = [];
    foreach ($items as $r) {
      $r = trim($r);
      if ($r === '') continue;
      
      if (filter_var($r, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4 | FILTER_FLAG_IPV6)) {
        $out[] = $r;
      } elseif (preg_match('#^(\d+\.\d+\.\d+\.\d+)/(\d+)$#', $r, $m)) {
        $ip = $m[1]; $prefix = (int)$m[2];
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4) && $prefix >= 0 && $prefix <= 32) {
          $out[] = $r;
        }
      } elseif (preg_match('#^([0-9a-fA-F:]+)/(\d+)$#', $r, $m)) {
        $ip = $m[1]; $prefix = (int)$m[2];
        if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6) && $prefix >= 0 && $prefix <= 128) {
          $out[] = $r;
        }
      }
    }
    $out = array_values(array_unique($out));
    $joined = implode(',', $out);
    return substr($joined, 0, 2000);
  }

  function get_user(mysqli $conn, int $id){
    $st = $conn->prepare("SELECT id,username,role,whitelist_ip,created_at FROM admin_users WHERE id=? LIMIT 1");
    $st->bind_param("i",$id); $st->execute();
    $r = $st->get_result()->fetch_assoc(); $st->close(); return $r ?: null;
  }

  /* Create staff - WITH LOGGING */
  if ($_SERVER['REQUEST_METHOD']==='POST' && ($_POST['action'] ?? '')==='create') {
    $u = sanitize_username($_POST['username'] ?? '');
    $p = (string)($_POST['password'] ?? '');
    $c = (string)($_POST['confirm'] ?? '');
    $w = sanitize_iplist($_POST['whitelist_ip'] ?? '');
    if ($u==='' || strlen($u)>50) { $err="Username wajib dan max 50 karakter."; }
    elseif (strlen($p)<6) { $err="Password minimal 6 karakter."; }
    elseif ($p !== $c) { $err="Konfirmasi password tidak sama."; }
    elseif ($w==='') { $err="Whitelist IP wajib diisi."; }
    else {
      $check = $conn->prepare("SELECT COUNT(*) as c FROM admin_users WHERE username=?");
      $check->bind_param("s",$u); $check->execute();
      if ($check->get_result()->fetch_assoc()['c'] > 0) {
        $check->close(); $err="Username sudah digunakan.";
      } else {
        $check->close();
        $hash = password_hash($p, PASSWORD_DEFAULT);
        $st = $conn->prepare("INSERT INTO admin_users(username,password,role,whitelist_ip) VALUES(?,?,'staff',?)");
        $st->bind_param("sss", $u, $hash, $w);
        if ($st->execute()) {
          $user_id = $conn->insert_id;
          $st->close();
          
          // Log activity
          $description = "Menambahkan admin staff baru: {$u}";
          $detail = [
            'user_id' => $user_id,
            'username' => $u,
            'role' => 'staff',
            'whitelist_ip' => $w
          ];
          log_admin_action('USER_ADD', $description, $detail);
          
          $ok = "Staff '{$u}' berhasil ditambahkan.";
        } else {
          $st->close(); $err = "Gagal menyimpan ke database.";
        }
      }
    }
  }

  /* Update row (username + whitelist, opsional reset password) - WITH LOGGING */
  if ($_SERVER['REQUEST_METHOD']==='POST' && ($_POST['action'] ?? '')==='update_all') {
    $id = (int)($_POST['id'] ?? 0);
    $u = sanitize_username($_POST['username'] ?? '');
    $w = sanitize_iplist($_POST['whitelist_ip'] ?? '');
    $np = (string)($_POST['new_password'] ?? '');
    $cp = (string)($_POST['confirm_password'] ?? '');
    
    if ($id<=0) { $err="ID tidak valid."; }
    elseif ($u==='' || strlen($u)>50) { $err="Username wajib dan max 50 karakter."; }
    elseif ($w==='') { $err="Whitelist IP wajib diisi."; }
    elseif ($np!=='' && strlen($np)<6) { $err="Password baru minimal 6 karakter."; }
    elseif ($np!=='' && $np!==$cp) { $err="Konfirmasi password baru tidak sama."; }
    else {
      // Get existing data for logging
      $existing = get_user($conn, $id);
      if (!$existing) {
        $err = "User tidak ditemukan.";
      } else {
        $check = $conn->prepare("SELECT COUNT(*) as c FROM admin_users WHERE username=? AND id!=?");
        $check->bind_param("si",$u,$id); $check->execute();
        if ($check->get_result()->fetch_assoc()['c'] > 0) {
          $check->close(); $err="Username sudah digunakan oleh user lain.";
        } else {
          $check->close();
          
          $changes = [];
          if ($existing['username'] !== $u) $changes['username'] = ['old' => $existing['username'], 'new' => $u];
          if ($existing['whitelist_ip'] !== $w) $changes['whitelist_ip'] = ['old' => $existing['whitelist_ip'], 'new' => $w];
          
          if ($np === '') {
            // update tanpa password
            $st = $conn->prepare("UPDATE admin_users SET username=?, whitelist_ip=? WHERE id=?");
            $st->bind_param("ssi", $u, $w, $id);
          } else {
            // update dengan password baru
            $hash = password_hash($np, PASSWORD_DEFAULT);
            $st = $conn->prepare("UPDATE admin_users SET username=?, password=?, whitelist_ip=? WHERE id=?");
            $st->bind_param("sssi", $u, $hash, $w, $id);
            $changes['password'] = ['old' => 'hidden', 'new' => 'updated'];
          }
          
          if ($st->execute()) {
            $st->close();
            
            // Include log helper
            require_once __DIR__ . "/includes/log_helper.php";
            
            // Enhanced logging
            $description = "Mengupdate admin staff: {$u} (ID: {$id})";
            $log_details = [
                'target_user' => [
                    'user_id' => $id,
                    'username' => $u,
                    'role' => 'staff'
                ],
                'changes' => $changes,
                'total_changes' => count($changes)
            ];
            
            log_admin_action('USER_UPDATE', $description, $log_details);
            
            $ok = "Data staff '{$u}' berhasil diupdate.";
          } else {
            $st->close(); $err = "Gagal mengupdate database.";
          }
        }
      }
    }
  }

  /* Delete staff - WITH LOGGING */
  if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    if ($id > 0) {
      // Get existing data for logging
      $existing = get_user($conn, $id);
      if ($existing) {
        $st = $conn->prepare("DELETE FROM admin_users WHERE id=? AND role='staff'");
        $st->bind_param("i", $id);
        if ($st->execute() && $st->affected_rows > 0) {
          $st->close();
          
          // Log activity
          $description = "Menghapus admin staff: {$existing['username']} (ID: {$id})";
          $detail = [
            'deleted_user' => $existing
          ];
          log_admin_action('USER_DELETE', $description, $detail);
          
          $ok = "Staff '{$existing['username']}' berhasil dihapus.";
        } else {
          $st->close(); $err = "Gagal menghapus atau data tidak ditemukan.";
        }
      } else {
        $err = "User tidak ditemukan.";
      }
    }
  }
}

/* List staff */
$staff = $conn->query("SELECT id,username,role,whitelist_ip,created_at FROM admin_users WHERE role='staff' ORDER BY username ASC");
?>
<!doctype html>
<html>
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Kelola Admin Staff</title>
  <link rel="stylesheet" href="../style_admin.css">
</head>
<body class="admin admin-users <?= $is_readonly ? 'readonly' : '' ?>">

<?php include 'partials/sidebar.php'; ?>

<!-- Mobile Toggle Button -->
<button class="mobile-toggle" onclick="toggleSidebar()">☰</button>

<!-- Main Content -->  
<div class="admin-main">
  <div class="wrap">
    <div class="page-title">
      <h2>👥 Kelola Admin Staff</h2>
      <div class="breadcrumb">
        <a href="dashboard.php">Dashboard</a> / <span>Admin Users</span>
      </div>
      <?php if ($is_readonly): ?>
        <div class="permission-info">
          <span class="permission-badge">📋 Read Only</span>
        </div>
      <?php endif; ?>
    </div>
    
    <?php if($err): ?>
      <div class="alert error">⚠️ <?= htmlspecialchars($err) ?></div>
    <?php endif; ?>
    
    <?php if($ok): ?>
      <div class="alert success">✅ <?= htmlspecialchars($ok) ?></div>
    <?php endif; ?>

    <?php if ($is_readonly): ?>
      <div class="alert info">
        ℹ️ Anda hanya dapat melihat daftar admin staff. Untuk menambah/edit/hapus, hubungi Master Admin.
      </div>
    <?php endif; ?>

    <!-- Add Staff Form -->
    <?php if (!$is_readonly): ?>
    <div class="card">
      <h3>➕ Tambah Admin Staff Baru</h3>
      <form method="post" autocomplete="off" class="admin-form">
        <input type="hidden" name="action" value="create">
        
        <div class="form-grid">
          <div class="form-group">
            <label>Username</label>
            <input type="text" name="username" required placeholder="Masukkan username staff">
          </div>
          
          <div class="form-group">
            <label>Password</label>
            <input type="password" name="password" autocomplete="new-password" required placeholder="Minimal 6 karakter">
          </div>
          
          <div class="form-group">
            <label>Konfirmasi Password</label>
            <input type="password" name="confirm" autocomplete="new-password" required placeholder="Ulangi password">
          </div>
          
          <div class="form-group">
            <label>Whitelist IP</label>
            <input type="text" name="whitelist_ip" required placeholder="203.0.113.10, 198.51.100.0/24">
            <small class="form-hint">Pisahkan dengan koma, boleh menggunakan CIDR notation</small>
          </div>
        </div>
        
        <div class="form-actions">
          <button type="submit" class="btn-primary">➕ Tambah Staff</button>
        </div>
      </form>
      
      <div class="info-text">
        💡 <strong>Contoh format IP:</strong> 203.0.113.10, 198.51.100.0/24, ::1 (IPv6). Staff hanya bisa login dari IP yang terdaftar.
      </div>
    </div>
    <?php endif; ?>

    <!-- Staff List -->
    <div class="card">
      <div class="table-header">
        <h3>👥 Daftar Admin Staff</h3>
        <div class="table-info">
          <?php
          $total_staff = $conn->query("SELECT COUNT(*) as total FROM admin_users WHERE role='staff'")->fetch_assoc()['total'];
          echo "Total: {$total_staff} staff";
          ?>
        </div>
      </div>
      
      <?php if ($staff->num_rows > 0): ?>
        <div class="staff-list">
          <?php while($u = $staff->fetch_assoc()): ?>
            <div class="staff-item">
              <div class="staff-info">
                <div class="staff-header">
                  <h4 class="staff-username">👤 <?= htmlspecialchars($u['username']) ?></h4>
                  <span class="staff-date">📅 <?= date('d/m/Y H:i', strtotime($u['created_at'])) ?></span>
                </div>
                <div class="staff-ip">
                  <strong>🔒 Whitelist IP:</strong> 
                  <code><?= htmlspecialchars($u['whitelist_ip'] ?? 'Tidak ada') ?></code>
                </div>
              </div>
              
              <?php if (!$is_readonly): ?>
              <form method="post" class="staff-form" autocomplete="off">
                <input type="hidden" name="action" value="update_all">
                <input type="hidden" name="id" value="<?= (int)$u['id'] ?>">
                
                <div class="form-grid">
                  <div class="form-group">
                    <label>Username</label>
                    <input type="text" name="username" value="<?= htmlspecialchars($u['username']) ?>" required>
                  </div>
                  
                  <div class="form-group">
                    <label>Whitelist IP</label>
                    <input type="text" name="whitelist_ip" value="<?= htmlspecialchars($u['whitelist_ip'] ?? '') ?>" 
                           placeholder="203.0.113.10, 198.51.100.0/24">
                  </div>
                  
                  <div class="form-group">
                    <label>Password Baru (Opsional)</label>
                    <input type="password" name="new_password" placeholder="Kosongkan jika tidak ingin mengubah">
                  </div>
                  
                  <div class="form-group">
                    <label>Konfirmasi Password</label>
                    <input type="password" name="confirm_password" placeholder="Ulangi password baru">
                  </div>
                </div>
                
                <div class="form-actions">
                  <button type="submit" class="btn-primary">💾 Simpan Perubahan</button>
                  <a href="admin_users.php?delete=<?= (int)$u['id'] ?>" 
                     class="btn-delete" 
                     onclick="return confirm('⚠️ Yakin ingin menghapus admin staff <?= htmlspecialchars($u['username']) ?>?')">
                    🗑️ Hapus
                  </a>
                </div>
              </form>
              <?php endif; ?>
            </div>
          <?php endwhile; ?>
        </div>
      <?php else: ?>
        <div class="empty-state">
          <div class="empty-icon">👥</div>
          <h4>Belum ada admin staff</h4>
          <?php if (!$is_readonly): ?>
          <p>Tambahkan admin staff pertama menggunakan form di atas.</p>
          <?php else: ?>
          <p>Belum ada staff yang terdaftar dalam sistem.</p>
          <?php endif; ?>
        </div>
      <?php endif; ?>
    </div>
  </div>
</div>

<script>
function toggleSidebar() {
  document.querySelector('.admin-sidebar').classList.toggle('mobile-open');
}
</script>

</body>
</html>