<?php
include_once "../config.php";
if (session_status() === PHP_SESSION_NONE) { session_start(); }

if (empty($_SESSION['logged'])) { 
    header("Location: login.php"); 
    exit; 
}

// Auto-check permission (Master otomatis bisa akses)
auto_check_page_permission();

// Include log helper
$log_helper_path = __DIR__ . "/includes/log_helper.php";
if (file_exists($log_helper_path)) {
    require_once $log_helper_path;
} else {
    error_log("PASARAN: Log helper NOT FOUND");
}

ob_start();

// ==== Util: resize image 120x120 (center-crop) ====
function resize_save_image(string $srcPath, string $dstPath, int $w=120, int $h=120): bool {
    // jika GD tidak tersedia, jangan fatal — biarkan caller fallback
    if (!extension_loaded('gd') || !function_exists('imagecreatetruecolor')) {
        return false;
    }

    $info = @getimagesize($srcPath);
    if(!$info) return false;

    $srcW = $info[0];
    $srcH = $info[1];
    $srcType = $info[2]; // IMAGETYPE_JPEG, IMAGETYPE_PNG, dll.

    // buat resource source
    $srcImg = null;
    switch($srcType) {
        case IMAGETYPE_JPEG: $srcImg = @imagecreatefromjpeg($srcPath); break;
        case IMAGETYPE_PNG:  $srcImg = @imagecreatefrompng($srcPath);  break;
        case IMAGETYPE_GIF:  $srcImg = @imagecreatefromgif($srcPath);  break;
        case IMAGETYPE_WEBP: 
            if (function_exists('imagecreatefromwebp')) {
                $srcImg = @imagecreatefromwebp($srcPath);
            }
            break;
        default: return false;
    }
    if(!$srcImg) return false;

    // hitung crop center
    $size = min($srcW, $srcH);
    $cropX = (int)(($srcW - $size) / 2);
    $cropY = (int)(($srcH - $size) / 2);

    // buat canvas target
    $destImg = imagecreatetruecolor($w, $h);
    if(!$destImg) {
        imagedestroy($srcImg);
        return false;
    }

    // jika PNG, preserve transparency
    if($srcType === IMAGETYPE_PNG) {
        imagesavealpha($destImg, true);
        $transparent = imagecolorallocatealpha($destImg, 0, 0, 0, 127);
        imagefill($destImg, 0, 0, $transparent);
    }

    // resample (crop + resize)
    $ok = imagecopyresampled(
        $destImg, $srcImg,
        0, 0,          // dst x,y
        $cropX, $cropY, // src x,y (cropped)
        $w, $h,        // dst w,h
        $size, $size   // src w,h (square crop)
    );

    if($ok) {
        // simpan ke file tujuan
        switch($srcType) {
            case IMAGETYPE_JPEG: $ok = imagejpeg($destImg, $dstPath, 85); break;
            case IMAGETYPE_PNG:  $ok = imagepng($destImg, $dstPath); break;
            case IMAGETYPE_GIF:  $ok = imagegif($destImg, $dstPath); break;
            case IMAGETYPE_WEBP: 
                if (function_exists('imagewebp')) {
                    $ok = imagewebp($destImg, $dstPath, 85);
                } else {
                    $ok = imagejpeg($destImg, $dstPath, 85);
                }
                break;
            default: $ok = false;
        }
    }

    imagedestroy($srcImg);
    imagedestroy($destImg);
    return $ok;
}

// Helper slug
if (!function_exists('slugify_name')) {
    function slugify_name(string $name): string {
        $slug = strtolower(trim($name));
        $slug = preg_replace('/[^a-z0-9\s-]/', '', $slug);
        $slug = preg_replace('/[\s-]+/', '-', $slug);
        return trim($slug, '-');
    }
}

// msg untuk notifikasi
$msg = null;

// ==== TAMBAH PASARAN - WITH LOGGING ====
if (isset($_POST['tambah'])) {
    $nama = trim($_POST['nama'] ?? '');
    $kode = trim($_POST['kode'] ?? '');
    $jam_tutup = trim($_POST['jam_tutup'] ?? '');
    $jam_result = trim($_POST['jam_result'] ?? '');
    
    if (empty($nama) || empty($kode) || empty($jam_tutup) || empty($jam_result)) {
        $msg = "Semua field wajib diisi!";
    } elseif (!isset($_FILES['logo']) || $_FILES['logo']['error'] !== UPLOAD_ERR_OK) {
        $msg = "Logo wajib diupload!";
    } else {
        // cek duplikat kode
        $check = $conn->prepare("SELECT COUNT(*) as c FROM pasaran WHERE kode = ?");
        $check->bind_param("s", $kode);
        $check->execute();
        $exists = $check->get_result()->fetch_assoc()['c'] > 0;
        $check->close();
        
        if ($exists) {
            $msg = "Kode pasaran '$kode' sudah digunakan!";
        } else {
            // handle upload logo
            $uploaded = $_FILES['logo'];
            $tmpPath = $uploaded['tmp_name'];
            $originalName = $uploaded['name'];
            $fileSize = $uploaded['size'];
            
            // validasi
            if ($fileSize > 2 * 1024 * 1024) { // 2MB
                $msg = "Ukuran file maksimal 2MB!";
            } else {
                $ext = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
                if (!in_array($ext, ['jpg','jpeg','png','gif','webp'])) {
                    $msg = "Format file harus JPG, PNG, GIF, atau WebP!";
                } else {
                    // buat nama file unik
                    $slug = slugify_name($nama);
                    $newName = $slug . '-' . time() . '.' . $ext;
                    
                    // pastikan folder ada
                    $uploadDir = '../img/pasaran/';
                    if (!is_dir($uploadDir)) {
                        mkdir($uploadDir, 0755, true);
                    }
                    
                    $targetPath = $uploadDir . $newName;
                    
                    // resize dan simpan
                    if (resize_save_image($tmpPath, $targetPath, 120, 120)) {
                        // insert ke DB
                        $stmt = $conn->prepare("INSERT INTO pasaran (nama, kode, logo, jam_tutup, jam_result) VALUES (?, ?, ?, ?, ?)");
                        $stmt->bind_param("sssss", $nama, $kode, $newName, $jam_tutup, $jam_result);
                        
                        if ($stmt->execute()) {
                            $pasaran_id = $conn->insert_id;
                            $stmt->close();
                            
                            // Log activity
                            $description = "Menambahkan pasaran baru: {$nama} (Kode: {$kode})";
                            $detail = [
                                'pasaran_id' => $pasaran_id,
                                'nama' => $nama,
                                'kode' => $kode,
                                'jam_tutup' => $jam_tutup,
                                'jam_result' => $jam_result,
                                'logo' => $newName
                            ];
                            require_once __DIR__."/includes/log_helper.php"; log_admin_action('PASARAN_ADD', $description, $detail);
                            
                            header("Location: admin_pasaran.php?ok=1");
                            exit;
                        } else {
                            $stmt->close();
                            unlink($targetPath); // hapus file jika gagal insert
                            $msg = "Gagal menyimpan ke database!";
                        }
                    } else {
                        $msg = "Gagal memproses gambar! Pastikan file valid.";
                    }
                }
            }
        }
    }
}

// ==== Update (Edit gabungan) - WITH LOGGING ====
if (isset($_POST['update_pasaran'])) {
    $id = (int)($_POST['id'] ?? 0);
    $nama = trim($_POST['nama'] ?? '');
    $kode = trim($_POST['kode'] ?? '');
    $jam_tutup = trim($_POST['jam_tutup'] ?? '');
    $jam_result = trim($_POST['jam_result'] ?? '');
    
    if ($id <= 0 || empty($nama) || empty($kode) || empty($jam_tutup) || empty($jam_result)) {
        $msg = "Data tidak valid!";
    } else {
        // Get existing data for logging
        $existing = $conn->query("SELECT * FROM pasaran WHERE id = {$id}")->fetch_assoc();
        
        // cek duplikat kode (selain diri sendiri)
        $check = $conn->prepare("SELECT COUNT(*) as c FROM pasaran WHERE kode = ? AND id != ?");
        $check->bind_param("si", $kode, $id);
        $check->execute();
        $exists = $check->get_result()->fetch_assoc()['c'] > 0;
        $check->close();
        
        if ($exists) {
            $msg = "Kode pasaran '$kode' sudah digunakan!";
        } else {
            $logoName = $existing['logo'] ?? ''; // keep existing logo by default
            
            // handle upload logo baru (opsional)
            if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
                $uploaded = $_FILES['logo'];
                $tmpPath = $uploaded['tmp_name'];
                $originalName = $uploaded['name'];
                $fileSize = $uploaded['size'];
                
                if ($fileSize > 2 * 1024 * 1024) {
                    $msg = "Ukuran file maksimal 2MB!";
                } else {
                    $ext = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
                    if (!in_array($ext, ['jpg','jpeg','png','gif','webp'])) {
                        $msg = "Format file harus JPG, PNG, GIF, atau WebP!";
                    } else {
                        $slug = slugify_name($nama);
                        $newName = $slug . '-' . time() . '.' . $ext;
                        $uploadDir = '../img/pasaran/';
                        $targetPath = $uploadDir . $newName;
                        
                        if (resize_save_image($tmpPath, $targetPath, 120, 120)) {
                            // hapus logo lama jika ada
                            if (!empty($logoName) && file_exists($uploadDir . $logoName)) {
                                unlink($uploadDir . $logoName);
                            }
                            $logoName = $newName;
                        } else {
                            $msg = "Gagal memproses gambar baru!";
                        }
                    }
                }
            }
            
            if (!$msg) {
                // update database
                $stmt = $conn->prepare("UPDATE pasaran SET nama=?, kode=?, logo=?, jam_tutup=?, jam_result=? WHERE id=?");
                $stmt->bind_param("sssssi", $nama, $kode, $logoName, $jam_tutup, $jam_result, $id);
                
                if ($stmt->execute()) {
                    $stmt->close();
                    
                    // Log activity
                    $description = "Mengupdate pasaran: {$nama} (ID: {$id})";
                    $changes = [];
                    if ($existing['nama'] !== $nama) $changes['nama'] = ['old' => $existing['nama'], 'new' => $nama];
                    if ($existing['kode'] !== $kode) $changes['kode'] = ['old' => $existing['kode'], 'new' => $kode];
                    if ($existing['jam_tutup'] !== $jam_tutup) $changes['jam_tutup'] = ['old' => $existing['jam_tutup'], 'new' => $jam_tutup];
                    if ($existing['jam_result'] !== $jam_result) $changes['jam_result'] = ['old' => $existing['jam_result'], 'new' => $jam_result];
                    if ($existing['logo'] !== $logoName) $changes['logo'] = ['old' => $existing['logo'], 'new' => $logoName];
                    
                    $detail = [
                        'pasaran_id' => $id,
                        'changes' => $changes
                    ];
                    log_admin_action('PASARAN_UPDATE', $description, $detail);
                    
                    header("Location: admin_pasaran.php?ok=1");
                    exit;
                } else {
                    $stmt->close();
                    $msg = "Gagal mengupdate database!";
                }
            }
        }
    }
}

// ==== Delete (Hapus Pasaran) - WITH LOGGING ====
if (isset($_GET['hapus'])) {
    $id = (int)$_GET['hapus'];
    if ($id > 0) {
        // Get existing data for logging
        $existing = $conn->query("SELECT * FROM pasaran WHERE id = {$id}")->fetch_assoc();
        
        if ($existing) {
            // hapus file logo jika ada
            $logoPath = '../img/pasaran/' . $existing['logo'];
            if (!empty($existing['logo']) && file_exists($logoPath)) {
                unlink($logoPath);
            }
            
            // hapus dari database
            $stmt = $conn->prepare("DELETE FROM pasaran WHERE id = ?");
            $stmt->bind_param("i", $id);
            
            if ($stmt->execute()) {
                $stmt->close();
                
                // Log activity
                $description = "Menghapus pasaran: {$existing['nama']} (Kode: {$existing['kode']})";
                $detail = [
                    'deleted_pasaran' => $existing
                ];
                log_admin_action('PASARAN_DELETE', $description, $detail);
                
                header("Location: admin_pasaran.php?ok=1");
                exit;
            } else {
                $stmt->close();
                $msg = "Gagal menghapus pasaran!";
            }
        } else {
            $msg = "Pasaran tidak ditemukan!";
        }
    }
}
?>

<!doctype html>
<html>
<head>
  <meta charset="utf-8">
  <title>Admin Pasaran - Admin</title>
  <link rel="stylesheet" href="../style_admin.css">
</head>
<body class="admin <?= is_master() ? 'role-master' : 'role-staff' ?>">

<?php include 'partials/sidebar.php'; ?>

<button class="mobile-toggle" onclick="toggleSidebar()">☰</button>

<div class="admin-main">
  <div class="wrap">
    <div class="page-title">
      <h2>🏆 Admin Pasaran</h2>
      <div class="breadcrumb">
        <a href="dashboard.php">Dashboard</a> > <span>Admin Pasaran</span>
      </div>
      <?php if (!is_master()): ?>
        <div class="permission-info">
          <span class="permission-badge">📋 Staff Permission</span>
        </div>
      <?php endif; ?>
    </div>

    <?php if (isset($_GET['ok'])): ?>
      <div class="ok">✅ Perubahan berhasil disimpan!</div>
    <?php endif; ?>
    
    <?php if (!empty($msg)): ?>
      <div class="err">⚠️ <?= htmlspecialchars($msg) ?></div>
    <?php endif; ?>

    <!-- Form Section -->
    <div class="form-section">
      <?php if (isset($_GET['edit'])): ?>
        <?php
        $edit_id = intval($_GET['edit']);
        $edit = $conn->query("SELECT id, nama, kode, logo, jam_tutup, jam_result FROM pasaran WHERE id=".$edit_id)->fetch_assoc();
        if ($edit):
        ?>
        <!-- Edit Form -->
        <div class="card">
          <h3>Edit Pasaran: <?= htmlspecialchars($edit['nama']) ?></h3>
          <form method="post" enctype="multipart/form-data" class="admin-form">
            <input type="hidden" name="id" value="<?= intval($edit['id']) ?>">
            
            <div class="form-grid">
              <div class="form-group">
                <label>Nama Pasaran</label>
                <input type="text" name="nama" required value="<?= htmlspecialchars($edit['nama']) ?>" placeholder="Contoh: SINGAPORE">
              </div>
              
              <div class="form-group">
                <label>Kode Pasaran</label>
                <input type="text" name="kode" required value="<?= htmlspecialchars($edit['kode']) ?>" placeholder="Contoh: SGP">
              </div>
              
              <div class="form-group">
                <label>Jam Tutup</label>
                <input type="time" name="jam_tutup" required value="<?= htmlspecialchars($edit['jam_tutup']) ?>">
              </div>
              
              <div class="form-group">
                <label>Jam Result</label>
                <input type="time" name="jam_result" required value="<?= htmlspecialchars($edit['jam_result']) ?>">
              </div>
            </div>
            
            <div class="form-group">
              <label>Logo Pasaran (Opsional)</label>
              <input type="file" name="logo" accept="image/*" class="file-input">
              <?php if (!empty($edit['logo'])): ?>
                <div class="current-logo">
                  <img src="../img/pasaran/<?= htmlspecialchars($edit['logo']) ?>" alt="Logo saat ini">
                  <span>Logo saat ini</span>
                </div>
              <?php endif; ?>
            </div>
            
            <div class="form-actions">
              <button type="submit" name="update_pasaran" class="btn-primary">💾 Simpan Perubahan</button>
              <a href="admin_pasaran.php" class="btn-secondary">❌ Batal</a>
            </div>
          </form>
        </div>
        <?php endif; ?>
      <?php else: ?>
        <!-- Add Form -->
        <div class="card">
          <h3>➕ Tambah Pasaran Baru</h3>
          <form method="post" enctype="multipart/form-data" class="admin-form">
            <div class="form-grid">
              <div class="form-group">
                <label>Nama Pasaran</label>
                <input type="text" name="nama" required placeholder="Contoh: SINGAPORE">
              </div>
              
              <div class="form-group">
                <label>Kode Pasaran</label>
                <input type="text" name="kode" required placeholder="Contoh: SGP">
              </div>
              
              <div class="form-group">
                <label>Jam Tutup</label>
                <input type="time" name="jam_tutup" required>
              </div>
              
              <div class="form-group">
                <label>Jam Result</label>
                <input type="time" name="jam_result" required>
              </div>
            </div>
            
            <div class="form-group">
              <label>Logo Pasaran</label>
              <input type="file" name="logo" accept="image/*" required class="file-input">
              <small class="form-hint">Format: JPG, PNG, GIF, WebP. Maksimal 2MB.</small>
            </div>
            
            <div class="form-actions">
              <button type="submit" name="tambah" class="btn-primary">➕ Tambah Pasaran</button>
            </div>
          </form>
        </div>
      <?php endif; ?>
    </div>

    <!-- Table Section -->
    <div class="card">
      <div class="table-header">
        <h3>📋 Daftar Pasaran</h3>
        <div class="table-info">
          <?php
          $total = $conn->query("SELECT COUNT(*) as total FROM pasaran")->fetch_assoc()['total'];
          echo "Total: {$total} pasaran";
          ?>
        </div>
      </div>
      
      <div class="table-responsive">
        <table class="admin-table">
          <thead>
            <tr>
              <th>Logo</th>
              <th>Nama</th>
              <th>Kode</th>
              <th>Jam Tutup</th>
              <th>Jam Result</th>
              <th>Aksi</th>
            </tr>
          </thead>
          <tbody>
            <?php
            $res = $conn->query("SELECT id, nama, kode, logo, jam_tutup, jam_result
                                 FROM pasaran
                                 ORDER BY STR_TO_DATE(jam_tutup, '%H:%i') ASC, nama ASC");
            while ($row = $res->fetch_assoc()):
                $logo_file = $row['logo'] ?? '';
                $path = "../img/pasaran/".$logo_file;
                $v = ($logo_file && is_file($path)) ? @filemtime($path) : 0;
            ?>
              <tr>
                <td class="logo-cell">
                  <?php if ($logo_file !== '' && is_file($path)): ?>
                    <img src="<?= htmlspecialchars($path) ?>?v=<?= $v ?>" alt="Logo <?= htmlspecialchars($row['nama']) ?>" class="table-logo">
                  <?php else: ?>
                    <div class="no-logo">📷</div>
                  <?php endif; ?>
                </td>
                <td class="pasaran-name"><?= htmlspecialchars($row['nama']) ?></td>
                <td class="pasaran-code"><?= htmlspecialchars($row['kode']) ?></td>
                <td class="time-cell">⏰ <?= htmlspecialchars($row['jam_tutup'] ?? '-') ?></td>
                <td class="time-cell">🎯 <?= htmlspecialchars($row['jam_result'] ?? '-') ?></td>
                <td class="actions-cell">
                  <a href="?edit=<?= intval($row['id']) ?>" class="btn-edit" title="Edit">✏️</a>
                  <a href="?hapus=<?= intval($row['id']) ?>" class="btn-delete" title="Hapus" onclick="return confirm('⚠️ Yakin ingin menghapus pasaran <?= htmlspecialchars($row['nama']) ?>?')">🗑️</a>
                </td>
              </tr>
            <?php endwhile; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<script>
function toggleSidebar() {
  document.querySelector('.admin-sidebar').classList.toggle('mobile-open');
}
</script>

</body>
</html>