/**
 * Smooth Navigation System
 * Versi 1.0
 */
(function() {
    'use strict';
    
    // Konfigurasi
    const config = {
        contentSelector: '#main-content',
        linkSelector: 'a:not([target="_blank"]):not([href^="#"]):not([href^="javascript:"]):not([data-no-ajax])',
        loadingClass: 'ajax-loading',
        activeClass: 'active',
        initClass: 'js-ajax-loaded',
        analyticsEnabled: false
    };
    
    // Cache
    const cache = {};
    
    // Fungsi untuk mendapatkan URL saat ini tanpa parameter dan hash
    function getCurrentURL() {
        return window.location.pathname;
    }
    
    // Fungsi untuk memuat halaman melalui AJAX
    function loadPage(url, pushState = true) {
        // Mulai loading
        document.body.classList.add(config.loadingClass);
        
        // Gunakan cache jika tersedia
        if (cache[url]) {
            updatePage(url, cache[url], pushState);
            return;
        }
        
        // Buat request
        fetch(url, {
            method: 'GET',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            },
            credentials: 'same-origin'
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.text();
        })
        .then(html => {
            // Parse HTML
            const parser = new DOMParser();
            const doc = parser.parseFromString(html, 'text/html');
            
            // Simpan ke cache
            cache[url] = {
                title: doc.title,
                content: doc.querySelector(config.contentSelector).innerHTML
            };
            
            // Update halaman
            updatePage(url, cache[url], pushState);
        })
        .catch(error => {
            console.error('Fetch error:', error);
            window.location.href = url; // Fallback ke navigasi tradisional
        });
    }
    
    // Fungsi untuk memperbarui konten halaman
    function updatePage(url, data, pushState) {
        // Update title
        document.title = data.title;
        
        // Update URL jika diperlukan
        if (pushState && window.history && window.history.pushState) {
            window.history.pushState({ url: url }, data.title, url);
        }
        
        // Update konten
        const contentElement = document.querySelector(config.contentSelector);
        if (contentElement) {
            contentElement.innerHTML = data.content;
        }
        
        // Update navigasi aktif
        updateActiveNav(url);
        
        // Jalankan script pada konten baru
        executeScripts();
        
        // Scroll ke atas
        window.scrollTo(0, 0);
        
        // Selesai loading
        document.body.classList.remove(config.loadingClass);
        
        // Kirim event konten dimuat
        window.dispatchEvent(new CustomEvent('contentLoaded', {
            detail: { url: url }
        }));
        
        // Google Analytics
        if (config.analyticsEnabled && typeof gtag === 'function') {
            gtag('config', 'YOUR-GA-ID', {
                'page_path': url
            });
        }
    }
    
    // Update navigasi aktif
    function updateActiveNav(url) {
        // Hapus semua status aktif sebelumnya
        document.querySelectorAll('.main-nav a.' + config.activeClass).forEach(link => {
            link.classList.remove(config.activeClass);
        });
        
        // Tambahkan status aktif ke link yang sesuai dengan URL saat ini
        document.querySelectorAll('.main-nav a').forEach(link => {
            if (link.pathname === url || 
               (link.pathname === '/' && url === '/index.php') || 
               (url.indexOf(link.pathname) === 0 && link.pathname !== '/')) {
                link.classList.add(config.activeClass);
            }
        });
    }
    
    // Jalankan script pada konten baru
    function executeScripts() {
        // Dapatkan scripts dari konten baru
        const scriptElements = document.querySelector(config.contentSelector).querySelectorAll('script');
        
        // Eksekusi scripts
        scriptElements.forEach(oldScript => {
            const newScript = document.createElement('script');
            
            // Salin atribut
            Array.from(oldScript.attributes).forEach(attr => {
                newScript.setAttribute(attr.name, attr.value);
            });
            
            // Salin konten
            newScript.textContent = oldScript.textContent;
            
            // Ganti script lama
            oldScript.parentNode.replaceChild(newScript, oldScript);
        });
        
        // Inisialisasi komponen
        initComponents();
    }
    
    // Inisialisasi komponen UI
    function initComponents() {
        // Inisialisasi toggle result
        document.querySelectorAll('.btn-mobile-toggle').forEach(button => {
            button.addEventListener('click', function() {
                const card = this.closest('.result-card');
                if (!card) return;
                
                const isShown = card.classList.toggle('show-extra');
                const openLabel = this.getAttribute('data-open-label') || 'Result Lengkap';
                const closeLabel = this.getAttribute('data-close-label') || 'Tutup Result';
                
                this.textContent = isShown ? closeLabel : openLabel;
                this.setAttribute('aria-expanded', isShown ? 'true' : 'false');
                
                const extraResults = card.querySelector('.extra-results');
                if (extraResults) {
                    extraResults.style.maxHeight = isShown ? '300px' : '0';
                    extraResults.style.opacity = isShown ? '1' : '0';
                    extraResults.style.visibility = isShown ? 'visible' : 'hidden';
                }
            });
        });
        
        // Inisialisasi modal history
        document.querySelectorAll('.btn-history').forEach(button => {
            button.addEventListener('click', function() {
                const pasaranId = this.getAttribute('data-pasaran-id');
                openHistoryModal('history.php?pasaran_id=' + pasaranId);
            });
        });
    }
    
    // Fungsi untuk tangani event klik
    function handleClick(e) {
        // Cari apakah klik pada link
        const link = e.target.closest(config.linkSelector);
        
        // Jika bukan link yang dimaksud, kembalikan
        if (!link) return;
        
        // Periksa apakah link internal
        if (link.hostname !== window.location.hostname) return;
        
        // Periksa apakah link ke halaman saat ini (dengan anchor)
        if (link.pathname === window.location.pathname && link.hash) return;
        
        // Cegah navigasi default
        e.preventDefault();
        
        // Muat halaman
        loadPage(link.href);
    }
    
    // Handler back/forward browser
    function handlePopState(e) {
        if (e.state && e.state.url) {
            loadPage(e.state.url, false);
        } else {
            loadPage(window.location.pathname, false);
        }
    }
    
    // Inisialisasi
    function init() {
        // Tunggu sampai DOM siap
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', setup);
        } else {
            setup();
        }
    }
    
    // Setup semua event listener
    function setup() {
        // Tambahkan kelas inisialisasi
        document.documentElement.classList.add(config.initClass);
        
        // Tangani klik pada link
        document.addEventListener('click', handleClick);
        
        // Tangani navigasi back/forward browser
        window.addEventListener('popstate', handlePopState);
        
        // Simpan state awal
        const initialUrl = getCurrentURL();
        window.history.replaceState({ url: initialUrl }, document.title, initialUrl);
        
        // Inisialisasi komponen
        initComponents();
    }
    
    // Mulai!
    init();
})();